PhysicsLab = class()

function PhysicsLab:init()
    self.bodies = {}
    self.joints = {}
    self.contacts = {}
    self.sensors = {}
    self.defaultGravity = physics.gravity()
    self.shapeMaker = ShapeMaker()
    self.jointMaker = JointMaker()
    self.drawer = ShapeDrawer()
    self.categoryManager = CategoryManager()
    self.shouldDraw = {["bodies"] = true, ["joints"] = true, ["contacts"] = true, ["touchMap"] = true}
    self.collideFunction = function(contact)
        if contact.state == BEGAN then
            self.contacts[contact.id] = contact
            sound(SOUND_HIT, 2643)
        elseif contact.state == MOVING then
            self.contacts[contact.id] = contact
        elseif contact.state == ENDED then
            self.contacts[contact.id] = nil
        end
    end
    self.touchMap = {}
    self.touchMapFunctions = {
        function(touchMap)
            local gain = 2.0
            local damp = 0.5
            for k,v in pairs(touchMap) do
                local worldAnchor = v.body:getWorldPoint(v.anchor)
                local touchPoint = v.tp
                local diff = touchPoint - worldAnchor
                local vel = v.body:getLinearVelocityFromWorldPoint(worldAnchor)
                v.body:applyForce( (1/1) * diff * gain - vel * damp, worldAnchor)
                
                line(touchPoint.x, touchPoint.y, worldAnchor.x, worldAnchor.y)
            end
        end
    }
end

-- Creating bodies:

function PhysicsLab:circleAt(x, y, radius)
    self:addBody(self.shapeMaker:circleAt(x, y, radius))
    return self.bodies[#self.bodies]
end

function PhysicsLab:polygonAt(x, y, pointsTable)
    self:addBody(self.shapeMaker:polygonAt(x, y, pointsTable))
    return self.bodies[#self.bodies]
end

function PhysicsLab:boxAt(x, y, w, h)
    self:addBody(self.shapeMaker:boxAt(x, y, w, h))
    return self.bodies[#self.bodies]
end

function PhysicsLab:randomPolygonAt(x, y, minSize, maxSize)
    self:addBody(self.shapeMaker:randomPolygonAt(x, y, minSize, maxSize))
    return self.bodies[#self.bodies]
end

function PhysicsLab:collidableWall(x1, y1, x2, y2)
    self:addBody(self.shapeMaker:collidableWall(x1, y1, x2, y2))
    return self.bodies[#self.bodies]
end

function PhysicsLab:CHAIN_SineWaveBetweenPoints(point1, point2, numCycles, waveHeight, segmentLength)
    self:addBody(self.shapeMaker:CHAIN_SineWaveBetweenPoints(point1, point2, numCycles, waveHeight, segmentLength))
    return self.bodies[#self.bodies]
end

function PhysicsLab:POLYGON_SineWaveBetweenPoints(point1, point2, numCycles, waveHeight, segmentLength)
    self:addBody(self.shapeMaker:POLYGON_SineWaveBetweenPoints(point1, point2, numCycles, waveHeight, segmentLength))
    return self.bodies[#self.bodies]
end

function PhysicsLab:makePetrieDish()  
    local groundLevel = WIDTH / 20
    local wallBotttomWidth = groundLevel * 0.55
    local outsideWallHeight = groundLevel * 11
    local insideWallHeight = outsideWallHeight * 0.7
    local wallTopWidth = wallBotttomWidth * 0.33
    local petrieDish = self.shapeMaker:polygonAt(0, 0, {
        vec2(0,0), --LL corner
        vec2(WIDTH,0), --LR corner
        vec2(WIDTH, outsideWallHeight), --outer R wall top corner
        vec2(WIDTH-wallTopWidth, outsideWallHeight), --inner R wall top corner 
        vec2(WIDTH-wallBotttomWidth, insideWallHeight), --inner R wall slope lower corner
        vec2(WIDTH-wallBotttomWidth, groundLevel), --inner R wall lower corner
        vec2(wallBotttomWidth, groundLevel), --inner L wall lower corner
        vec2(wallBotttomWidth, insideWallHeight), --inner L wall slope lower corner
        vec2(wallTopWidth, outsideWallHeight), --inner R wall slope lower corner
        vec2(0, outsideWallHeight) --outer L wall top corner
    })
    petrieDish.type = STATIC
    petrieDish.name = "petrieDish"
    self:addBody(petrieDish)
    return petrieDish
end

function PhysicsLab:addBody(body)
    table.insert(self.bodies, body)
end

function PhysicsLab:addJoint(joint)
    table.insert(self.joints, joint)
end

-- Rotating, pushing, and pulling:

function PhysicsLab:rotateBodyToFace(targetPoint, body, speed)
    local targetAngle = self:angleBetweenPoints(targetPoint, body.position)
    body.rotationTask = {angle = targetAngle, speed = speed}
end

function PhysicsLab:rotateBodyToFaceAwayFrom(targetPoint, body, speed)
    local targetAngle = self:angleBetweenPoints(body.position, targetPoint)
    body.rotationTask = {angle = targetAngle, speed = speed}
end

function PhysicsLab:angleBetweenPoints(point1, point2)
    local deltaX = point2.x - point1.x
    local deltaY = point1.y - point2.y
    local radsToAngleRatio = 180 / math.pi
    return math.atan(deltaX, deltaY) * radsToAngleRatio
end

function PhysicsLab:pullBodyTowards(targetPoint, body, speed, stopAtDistance)
    if body.pushTask then
        local task = body.pushTask
        task.anchor:destroy()
        task.pusher:destroy()
        body.pushTask = nil
    end
    if body.pullTask then
        local task = body.pullTask
        task.anchor:destroy()
        task.puller:destroy()
    end
    local anchorBody = self:circleAt(targetPoint.x, targetPoint.y, 0.001)
    anchorBody.type = STATIC
    local puller = self:barBetweenBodies(anchorBody, body, anchorBody.position, body.position)
    body.pullTask = {target = targetPoint, speed = speed, stop = stopAtDistance, puller = puller, anchor = anchorBody}
end

function PhysicsLab:pushBodyAwayFrom(sourcePoint, body, speed, stopAtDistance)
    if body.pushTask then
        local task = body.pushTask
        task.anchor:destroy()
        task.pusher:destroy()
    end
    if body.pullTask then
        local task = body.pullTask
        task.anchor:destroy()
        task.puller:destroy()
        body.pullTask = nil
    end
    local anchorBody = self:circleAt(sourcePoint.x, sourcePoint.y, 0.001)
    anchorBody.type = STATIC
    local pusher = self:barBetweenBodies(anchorBody, body, anchorBody.position, body.position)
    body.pushTask = {source = sourcePoint, speed = speed, stop = stopAtDistance, pusher = pusher, anchor = anchorBody}
end

-- Creating joints (including motorized joint):

function PhysicsLab:pointThatTwoBodiesRevolveAround(x, y, bodyA, bodyB)
    self:addJoint(self.jointMaker:pointThatTwoBodiesRevolveAround(x, y, bodyA, bodyB))
    return self.joints[#self.joints]
end

function PhysicsLab:barBetweenBodies(bodyA, bodyB, anchorOfBodyA, anchorOfBodyB)
    self:addJoint(self.jointMaker:barBetweenBodies(bodyA, bodyB, anchorOfBodyA, anchorOfBodyB))
    return self.joints[#self.joints]
end

function PhysicsLab:keepBodiesOnRailDefinedByPoints(bodyA, bodyB, axisPoint1, axisPoint2)
    self:addJoint(self.jointMaker:keepBodiesOnRailDefinedByPoints(bodyA, bodyB, axisPoint1, axisPoint2))
    return self.joints[#self.joints]
end

function PhysicsLab:weldBodiesTogether(bodyA, bodyB, weldPointBodyA, weldPointBodyB)
    self:addJoint(self.jointMaker:weldBodiesTogether(bodyA, bodyB, weldPointBodyA, weldPointBodyB))
    return self.joints[#self.joints]
end

function PhysicsLab:motorizedJoint(bodyA, bodyB, locationOfJoint, maxTorque)
    self:addJoint(self.jointMaker:motorizedJoint(bodyA, bodyB, locationOfJoint, maxTorque))
    return self.joints[#self.joints]
end

-- Collision detection:

function PhysicsLab:makeSensor(body, collideFunction)
    body.sensor = true
    body.collide = collideFunction
    table.insert(self.sensors, body)
end

function PhysicsLab:collide(contact)
    if self.collideFunction then self.collideFunction(contact) end
    for _, sensor in ipairs(self.sensors) do
        sensor.collide(contact)
    end
end

-- Intersection querying:

function PhysicsLab:cornersBounding(body)
    return self.drawer:cornersBounding(body)
end

function PhysicsLab:applyToResultsIntersectingLine(startPoint, endPoint, functionToApply, categoryNumbersToIgnore)
    categoryNumbersToIgnore = categoryNumbersToIgnore or {}
    local tablesDescribingResults = physics.raycastAll(startPoint, endPoint, table.unpack(categoryNumbersToIgnore))
    if functionToApply then
        functionToApply(tablesDescribingResults)
    end
    return tablesDescribingResults
end

function PhysicsLab:applyToAABBRectsIntersecting(lowerLeftPoint, upperRightPoint, functionToApply)
    local bodiesWithIntersectingRects = physics.queryAABB(lowerLeftPoint, upperRightPoint)
    if functionToApply then
        functionToApply(bodiesWithIntersectingRects)
    end
    return bodiesWithIntersectingRects
end

function PhysicsLab:applyToBodiesOverlapping(body, functionToApply)
    local overlaps = {}
    for _, storedBody in ipairs(self.bodies) do
        if body:testOverlap(storedBody) then
            table.insert(overlaps, storedBody)
        end
    end
    functionToApply(overlaps)
    return overlaps
end

-- Drawing and touch handling:

function PhysicsLab:shouldDrawAll(intendedStatus)
    --this is a setter if called with an argument, and a getter if not
    if intendedStatus then
        self.shouldDraw.bodies = intendedStatus
        self.shouldDraw.joints = intendedStatus
        self.shouldDraw.contacts = intendedStatus
        self.shouldDraw.touchMap = intendedStatus
    else
        return self.shouldDraw.bodies 
        and self.shouldDraw.joints 
        and self.shouldDraw.contacts 
        and self.shouldDraw.touchMap
    end
end

function PhysicsLab:draw()
    if self:shouldDrawAll() then
        self.drawer:drawThese(self.bodies, self.joints, self.contacts, self.touchMap)
    else
        if self.shouldDraw.bodies then
            self.drawer:drawBodies(self.bodies)
        end
        if self.shouldDraw.joints then
            self.drawer:drawJoints(self.joints)
        end
        if self.shouldDraw.contacts then
            self.drawer:drawContacts(self.contacts)
        end
        if self.shouldDraw.touchMap then
            self.drawer:drawTouchMap(self.touchMap)
        end
    end
    for _, touchMapFunction in ipairs(self.touchMapFunctions) do
        touchMapFunction(self.touchMap)
    end
    for _, body in ipairs(self.bodies) do
        if body.rotationTask then   
            --rotation code adapted from dave1707's rotation demo        
            local task = body.rotationTask 
            local vel = 0  
            if task.angle % 360 ~= body.angle % 360 then
                vel= (task.angle % 360) - (body.angle % 360)
                body.angularVelocity=vel*task.speed
            else
                body.angularVelocity=0
                body.rotationTask = nil
            end
        end
        if body.pullTask then
            local task = body.pullTask
            local stoppingDistance = task.stop or 0
            local currentDistance = body.position:dist(task.target)
            if currentDistance > stoppingDistance then
                local leftToTravel = currentDistance - stoppingDistance
                local movement = task.speed
                if movement > leftToTravel then movement = leftToTravel end
                task.puller.length = task.puller.length - movement
            else
                task.puller:destroy()
                task.anchor:destroy()
                body.pullTask = nil
            end
        end
        if body.pushTask then
            local task = body.pushTask
            local currentDistance = body.position:dist(task.source)
            local stoppingDistance = task.stop or currentDistance * 2
            if currentDistance < stoppingDistance then
                local leftToTravel = stoppingDistance - currentDistance
                local movement = task.speed
                if movement > leftToTravel then movement = leftToTravel end
                task.pusher.length = task.pusher.length + movement
            else
                task.pusher:destroy()
                task.anchor:destroy()
                body.pushTask = nil
            end
        end
    end
end

function PhysicsLab:drawAllAABB(drawColor, bodiesToIgnore)
    self.drawer:drawAllAABB(drawColor, bodiesToIgnore)
end

function PhysicsLab:touched(touch)
    local touchPoint = vec2(touch.x, touch.y)
    if touch.state == BEGAN then
        for i,body in ipairs(self.bodies) do
            if body.type == DYNAMIC and body:testPoint(touchPoint) then
                self.touchMap[touch.id] = {tp = touchPoint, body = body, anchor = body:getLocalPoint(touchPoint)} 
                return true
            end
        end
    elseif touch.state == MOVING and self.touchMap[touch.id] then
        self.touchMap[touch.id].tp = touchPoint
        return true
    elseif touch.state == ENDED and self.touchMap[touch.id] then
        self.touchMap[touch.id] = nil
        return true;
    end
    return false
end

function PhysicsLab:destroy(body)
    removeIfIn(self.bodies, body)
    removeIfIn(self.joints, body)
    removeIfIn(self.sensors, body)
    self.categoryManager:removeFromManager(body)
    body:destroy()
end

function PhysicsLab:clear()  
    for i,body in ipairs(self.bodies) do
        body:destroy()
    end    
    for i,joint in ipairs(self.joints) do
        joint:destroy()
    end      
    for i,sensor in ipairs(self.sensors) do
        sensor:destroy()
    end      
    self.bodies = {}
    self.joints = {}
    self.contacts = {}
    self.touchMap = {}
    self.sensors = {}
end
